from typing import List

from qgis.core import QgsCoordinateReferenceSystem
from qgis.gui import QgsProjectionSelectionWidget
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QDoubleValidator, QIcon
from qgis.PyQt.QtWidgets import (
    QCheckBox,
    QDialog,
    QDialogButtonBox,
    QFormLayout,
    QGridLayout,
    QGroupBox,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QTableWidget,
    QTableWidgetItem,
    QVBoxLayout,
)

from rana_qgis_plugin.constant import PLUGIN_NAME
from rana_qgis_plugin.utils_api import get_filename_from_attachment_url


class ResultBrowser(QDialog):
    def __init__(self, parent, results: dict, scenario_crs: str, pixel_size: float):
        super().__init__(parent)
        self.setWindowTitle(PLUGIN_NAME)
        self.setMinimumWidth(400)
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        self.selected_results = []
        self.selected_nodata = None
        self.selected_pixelsize = None
        self.selected_crs = None

        self.download_raw_data_bx = QCheckBox("Download simulation results", self)
        self.download_raw_data_bx.setChecked(True)
        layout.addWidget(self.download_raw_data_bx)

        results_group = QGroupBox("Download post-processing results", self)
        results_group.setLayout(QGridLayout())

        postprocessed_rasters_group = QGroupBox("Generate raster results", self)
        postprocessed_rasters_group.setLayout(QGridLayout())

        self.results_table = QTableWidget(self)
        results_group.layout().addWidget(self.results_table)
        self.results_table.setColumnCount(2)
        self.results_table.verticalHeader().setVisible(False)
        self.results_table.horizontalHeader().setStretchLastSection(True)
        self.results_table.setHorizontalHeaderLabels(["Type", "File name"])

        self.postprocessed_rasters_table = QTableWidget(self)
        postprocessed_rasters_group.layout().addWidget(self.postprocessed_rasters_table)
        self.postprocessed_rasters_table.setColumnCount(2)
        self.postprocessed_rasters_table.verticalHeader().setVisible(False)
        self.postprocessed_rasters_table.horizontalHeader().setStretchLastSection(True)
        self.postprocessed_rasters_table.setHorizontalHeaderLabels(
            ["Type", "File name"]
        )

        inputs_group = QGroupBox("Generated raster result settings", self)
        inputs_form = QFormLayout(self)
        inputs_group.setLayout(inputs_form)

        self.no_data_box = QLineEdit(self)
        self.pixelsize_box = QLineEdit(self)
        self.crs_select_box = QgsProjectionSelectionWidget(self)

        self.no_data_box.setValidator(
            QDoubleValidator(
                bottom=-1000000.0,
                top=1000000.0,
                decimals=1,
                notation=QDoubleValidator.StandardNotation,
            )
        )
        self.pixelsize_box.setValidator(
            QDoubleValidator(
                bottom=0.00001,
                top=1000000.00000,
                decimals=5,
                notation=QDoubleValidator.StandardNotation,
            )
        )

        self.no_data_box.setText("-9999.0")
        self.pixelsize_box.setText(f"{pixel_size:.5f}")
        self.crs_select_box.setCrs(QgsCoordinateReferenceSystem(scenario_crs))

        inputs_form.addRow("NO DATA value:", self.no_data_box)
        inputs_form.addRow("Pixel size:", self.pixelsize_box)
        inputs_form.addRow("CRS:", self.crs_select_box)

        # only show raster download options if raster is selected
        self.raster_selected = False

        def check_raster_selected():
            select_states = [
                self.postprocessed_rasters_table.item(i, 0).checkState()
                for i in range(self.postprocessed_rasters_table.rowCount())
            ]
            any_selected = (
                any(state == Qt.CheckState.Checked for state in select_states)
                and len(select_states) > 0
            )
            self.no_data_box.setEnabled(any_selected)
            self.pixelsize_box.setEnabled(any_selected)
            self.crs_select_box.setEnabled(any_selected)

        self.postprocessed_rasters_table.cellChanged.connect(check_raster_selected)

        postprocessed_rasters_group.layout().addWidget(inputs_group)
        always_checked = ["max water depth (file)"]
        excluded_results = ["raw 3di output", "grid administration", "3di bathymetry"]
        for result in [r for r in results if r.get("attachment_url")]:
            if result["name"].lower() in excluded_results:
                continue
            self.results_table.insertRow(self.results_table.rowCount())
            type_item = QTableWidgetItem(result["name"])
            type_item.setFlags(
                Qt.ItemFlag.ItemIsEnabled | Qt.ItemFlag.ItemIsUserCheckable
            )
            check_state = (
                Qt.CheckState.Checked
                if result["name"].lower() in always_checked
                else Qt.CheckState.Unchecked
            )
            type_item.setCheckState(check_state)
            type_item.setData(Qt.ItemDataRole.UserRole, int(result["id"]))

            file_name = get_filename_from_attachment_url(result["attachment_url"])

            file_name_item = QTableWidgetItem(file_name)
            file_name_item.setFlags(Qt.ItemFlag.ItemIsEnabled)
            self.results_table.setItem(self.results_table.rowCount() - 1, 0, type_item)
            self.results_table.setItem(
                self.results_table.rowCount() - 1, 1, file_name_item
            )

        # timeseries rasters
        excluded_rasters = ["depth-dtri", "rain-quad", "s1-dtri"]

        for i, result in enumerate(
            [
                r
                for r in results
                if r.get("raster_id") and r.get("code") not in excluded_rasters
            ]
        ):
            self.postprocessed_rasters_table.insertRow(
                self.postprocessed_rasters_table.rowCount()
            )
            type_item = QTableWidgetItem(result["name"])
            type_item.setFlags(
                Qt.ItemFlag.ItemIsEnabled | Qt.ItemFlag.ItemIsUserCheckable
            )
            type_item.setCheckState(Qt.CheckState.Unchecked)
            type_item.setData(Qt.ItemDataRole.UserRole, int(result["id"]))

            file_name = result["code"]
            file_name_item = QTableWidgetItem(file_name)
            file_name_item.setFlags(Qt.ItemFlag.ItemIsEnabled)
            self.postprocessed_rasters_table.setItem(i, 0, type_item)
            self.postprocessed_rasters_table.setItem(i, 1, file_name_item)

        # When Lizard post-processing is still running, results is empty and only raw data can be downloaded
        if len(results) == 0:
            # show a warning
            icon_label = QLabel()
            icon_label.setPixmap(
                QIcon(":/images/themes/default/mIconWarning.svg").pixmap(16, 16)
            )
            icon_label.setAlignment(Qt.AlignVCenter)
            warning_label = QLabel(
                "Post-processing results are not available because they are still being processed."
            )
            warning_label.setTextFormat(Qt.RichText)
            warning_layout = QHBoxLayout()
            warning_layout.addWidget(icon_label)
            warning_layout.addWidget(warning_label)
            warning_layout.setAlignment(Qt.AlignVCenter)
            layout.addLayout(warning_layout)
            # disable raster settings
            self.no_data_box.setEnabled(False)
            self.pixelsize_box.setEnabled(False)
            self.crs_select_box.setEnabled(False)
            # check download raw data by default

        self.results_table.resizeColumnsToContents()
        layout.addWidget(results_group)

        self.postprocessed_rasters_table.resizeColumnsToContents()
        layout.addWidget(postprocessed_rasters_group)

        buttonBox = QDialogButtonBox(
            QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel
        )
        buttonBox.accepted.connect(self.accept)
        buttonBox.rejected.connect(self.reject)
        layout.addWidget(buttonBox)

    def get_selected_results(self) -> List[int]:
        return (
            self.selected_results,
            self.selected_nodata,
            self.selected_pixelsize,
            self.selected_crs,
        )

    def get_download_raw_result(self) -> bool:
        return self.download_raw_data_bx.isChecked()

    def accept(self) -> None:
        self.selected_results = []
        for r in range(self.results_table.rowCount()):
            name_item = self.results_table.item(r, 0)
            if name_item.checkState() == Qt.CheckState.Checked:
                id = int(name_item.data(Qt.ItemDataRole.UserRole))
                self.selected_results.append(id)

        for r in range(self.postprocessed_rasters_table.rowCount()):
            name_item = self.postprocessed_rasters_table.item(r, 0)
            if name_item.checkState() == Qt.CheckState.Checked:
                id = int(name_item.data(Qt.ItemDataRole.UserRole))
                self.selected_results.append(id)

        self.selected_nodata = float(self.no_data_box.text())
        self.selected_pixelsize = float(self.pixelsize_box.text())
        self.selected_crs = self.crs_select_box.crs().authid()

        return super().accept()
